package com.sap.idm.vds.connector.samplefile.impl;

import java.io.*;
import java.util.*;
import java.util.regex.*;

/**
 * Class to filter results
 * @author I054742
 *
 */
public class FileFilter implements FilenameFilter {
	
	/**
	 * Constructor
	 */
    public FileFilter () {
    }
    
    
    /**
     * Checks if the name of a file matches with the regular expression given by 'filter'
     * @param file -> The file where the name will be taken from
     * @param filter -> Regular expression
     * @return -> Returns true if the file name matches with the regular expression given in 'filter' 
     */
    public boolean accept(File file, String filter) {
    	
        return this.stringFilter(filter, file.getName());
    }
    
    
    /**
     * Checks if there is some attribute which name and value match with the regular expression given in 'filter'
     * @param attrList -> List of attributes (names and values)
     * @param filter -> Regular expression
     * @return -> Returns true if there is some attribute which 
     * 			  name and value match with the regular expression given in 'filter'
     */
    public boolean accept (HashMap attrList, String filter) {
    	
    	/* Prepares the filter for the format used in this class */
    	filter=this.cleanTrailingAndLeadingSpaces(filter);
    	filter = this.cleanParenthesis(filter);
    	
		String [] splitted = filter.split("=");
		/* If the filter has not the expected format then a false value is returned */
		if (splitted.length<2) return false;
		
		/* Creates the filter for both the name of attributes and the values of them */
		String attrNameFilter = splitted[0];
		String attrValueFilter = splitted[1];
		
		for (int i=2; i<splitted.length; i++) attrValueFilter+="="+splitted[i];
		
		attrNameFilter = this.cleanTrailingAndLeadingSpaces(attrNameFilter);
		attrValueFilter = this.cleanTrailingAndLeadingSpaces(attrValueFilter);
		
		attrNameFilter = attrNameFilter.toLowerCase();
		attrValueFilter = attrValueFilter.toLowerCase();
		
		/* Checks if there is some attribute which matches with both the filter for name and value */
		for (Iterator it = attrList.keySet().iterator(); it.hasNext(); ) {
    		
    		String attrName = (String)it.next();
    		if (this.stringFilter(attrName, attrNameFilter)==false) continue;
    		String attrValue = ((String)attrList.get(attrName)).toLowerCase();
			boolean accepted = this.stringFilter(attrValue,attrValueFilter);
			/* If finds some attribute which matches by both name and value then a true value is returned */
			if (accepted) return true;
    	}
		/* If does not find any attribute which matches, then it returns false */
		return false;
    }
    
    
    /**
     * Removes the trailing and leading spaces of a string
     * @param str -> The string to be treated
     * @return -> 'str' without trailing and leading spaces
     */
    private String cleanTrailingAndLeadingSpaces (String str) {
    	
    	/* If the string is null or has no chars then the result is the same string */
    	if (str==null) return null;
    	if (str.length()==0) return str;
    	int i=0, j=str.length()-1;
    	/* Removes the trailing and leading spaces */
    	while (i<str.length() && str.charAt(i)==' ') i++;
    	while (j>i && str.charAt(j)==' ') j--;
    	return str.substring(i, Math.min(j+1, str.length()));
    }
    
    
    /**
     * Removes parenthesis in the beginning and end of a String
     * @param str -> The string to be treated
     * @return -> 'str' without trailing and leading spaces
     */
    private String cleanParenthesis (String str) {
    	
    	/* If the string is null or has no chars then the result is the same string */
    	if (str==null) return null;
    	if (str.length()==0) return str;
    	/* Removes the parenthesis */
    	if (str.charAt(0)=='(') str=str.substring(1);
    	if (str.length()==0) return str;
    	if (str.charAt(str.length()-1)==')') str=str.substring(0, str.length()-1);
    	return str;
    }
    
    
    /**
     * Checks if the string 'str' matches with the regular expression 'filter'
     * @param str -> The string to be checked against the regular expression 'filter'
     * @param filter -> The regular expression
     * @return -> True if 'str' matches with the regular expression 'filter',
     * 			  false otherwise
     */
    private boolean stringFilter (String str, String filter) {
    	
    	/* Compile a pattern to be replaced in the filter */
    	Pattern pattern = Pattern.compile("\\*");
    	/* Replace in filter all the occurrences of '*' by '.*' in order to prepare 
    	 * the correct syntax for the matching of the string against the regular expression */
    	filter=pattern.matcher(filter).replaceAll(".*");
    	/* Checks if the string matches with the regular expression */
    	return Pattern.matches(filter,str);
    }
}
